package org.msh.etbm.services.mobile.sync;

import org.apache.commons.beanutils.PropertyUtils;
import org.jboss.seam.Component;
import org.msh.etbm.services.mobile.SyncDataConsumer;
import org.msh.etbm.services.mobile.model.*;
import org.msh.tb.cases.CaseEditingHome;
import org.msh.tb.cases.CaseHome;
import org.msh.tb.entities.*;
import org.msh.tb.entities.enums.*;
import org.msh.tb.login.UserSession;
import org.msh.tb.ng.entities.TbCaseNG;
import org.msh.utils.date.Period;

import javax.persistence.EntityManager;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Created by rmemoria on 5/9/17.
 */
public class CaseDataConsumer implements SyncDataConsumer<TbCaseDataRequest, Integer> {

    @Override
    public Integer consume(Tbunit unit, TbCaseDataRequest data) {
        CaseEditingHome caseEditingHome = (CaseEditingHome)Component.getInstance("caseEditingHome");
        CaseHome caseHome = (CaseHome)Component.getInstance("caseHome");
        caseHome.clearInstance();


        // recover the TB case (or create a new one)
        TbCaseNG tbcase = data.getServerId() != null ?
                recoverCase(unit, data) :
                initNewCase(unit, data);

        Patient patient = tbcase.getPatient();

        // set the patient data
        patient.setName(data.getName());
        patient.setMiddleName(data.getMiddleName());
        patient.setLastName(data.getLastName());
        patient.setBirthDate(data.getBirthDate());
        patient.setMotherName(data.getMotherName());
        patient.setGender(data.getGender());
        patient.setSecurityNumber(data.getSecurityNumber());

        EntityManager em = getEntityManager();
        em.persist(patient);

        tbcase.setClassification(data.getClassification());
        tbcase.setState(data.getState());
        tbcase.setPatientType(data.getPatientType());
        tbcase.setDiagnosisDate(data.getDiagnosisDate());
        tbcase.setDiagnosisType(data.getDiagnosisType());
        tbcase.setInfectionSite(data.getInfectionSite());
        tbcase.setPulmonaryType(getFieldValue(data.getPulmonaryId()));
        tbcase.setExtrapulmonaryType(getFieldValueComponent(data.getExtrapulmonaryId()));
        tbcase.setExtrapulmonaryType2(getFieldValueComponent(data.getExtrapulmonary2Id()));

        /*
         * This is a palliative solution, as the field healthunitRegistrationDate was removed from server data model.
         * In mobile, for confirmed DSTB cases the registration date is being inputted on tbcase.healthunitRegistrationDate.
         *
         * After removing the field healthunitRegistrationDate from the server model, it is expected that the date inputted
         * manually by the mobile user (tbcase.healthunitRegistrationDate) will be the date set on the server tbcase.registrationDate.
         *
         * TODO: remove the healthunitRegistrationDate field from mobile
         *
         * The if statement bellow will need to continue here, even after resolving the TO DO above. It will can only be removed
         * when all mobile receive the new version that attend the TO DO item above.
         */
        if (data.getClassification().equals(CaseClassification.TB)
                && data.getDiagnosisType().equals(DiagnosisType.CONFIRMED)
                && data.getHealthunitRegistrationDate() != null) {
            tbcase.setRegistrationDate(data.getHealthunitRegistrationDate());
        } else {
            tbcase.setRegistrationDate(data.getRegistrationDate());
        }

        if(tbcase.getDiagnosisType().equals(DiagnosisType.SUSPECT)) {
            tbcase.setSuspectClassification(tbcase.getClassification());
        }

        tbcase.setTreatmentPeriod(new Period());
        tbcase.getTreatmentPeriod().setIniDate(data.getIniTreatmentDate());
        tbcase.getTreatmentPeriod().setEndDate(data.getEndTreatmentDate());
        tbcase.setIniContinuousPhase(data.getIniContinuousPhase());
        tbcase.setOutcomeDate(data.getOutcomeDate());
        tbcase.setRegimen(getRegimenById(data.getRegimenId()));
        tbcase.setRegimenIni(getRegimenById(data.getRegimenIniId()));
        tbcase.setTbRegistrationNumber(data.getTbRegistrationNumber());
        tbcase.setTreatmentCategory(data.getTreatmentCategory());
        tbcase.setAge(data.getAge());

        Address address = new Address();
        address.setAddress(data.getAddress());
        address.setAdminUnit(getAdminUnitById(data.getAdminUnitId()));
        address.setZipCode(data.getZipCode());
        tbcase.setNotifAddress(address);
        if (tbcase.getDiagnosisType().equals(DiagnosisType.SUSPECT) && tbcase.getNotifAddress().getAdminUnit() == null) {
            tbcase.getNotifAddress().setAdminUnit(unit.getAdminUnit());
        }

        tbcase.setPhoneNumber(data.getPhoneNumber());
        tbcase.setMobileNumber(data.getMobileNumber());
        tbcase.setEmailAddress(data.getEmailAddress());

        tbcase.setHivPosition(data.getHivPosition());
        tbcase.setHivPositionDetail(data.getHivPositionDetail());

        tbcase.setIntakeAntiTBDrugs(data.getIntakeAntiTBDrugs() == Boolean.TRUE ? YesNoType.YES : YesNoType.NO);
        tbcase.setIntakeAntiTBDrugsDuration(data.getIntakeAntiTBDrugsDuration());
        tbcase.setSourceReferral(getFieldValueComponent(data.getSourceReferralId()));
        tbcase.setDotProvider(getFieldValueComponent(data.getDotProvider()));
        tbcase.setMaritalStatus(getFieldValueComponent(data.getMaritalStatus()));
        tbcase.setOccupation(getFieldValueComponent(data.getOccupation()));
        tbcase.setDrugResistanceType(data.getDrugResistanceType());
        tbcase.setMovedSecondLineTreatment(data.isMovedSecondLineTreatment());
        tbcase.setNationality(data.getNationality());

        // save the case
        caseHome.setInstance(tbcase);
        updateTreatmentHealthUnit(tbcase);
        updatePrescribedMedicines(tbcase, data.getPrescribedMedicines());
        updatePrevTBTreat(tbcase, data.getPrevTBTreatments());
        caseEditingHome.updatePatientAge();
        caseHome.persist();
        caseHome.updateCaseTags();

        Integer caseId = (Integer)caseHome.getId();

        // just return the id if it is a new case
        return data.getServerId() == null ? caseId : null;
    }

    /**
     * Recover a case by its server ID
     * @param unit
     * @param data
     * @return
     */
    private TbCaseNG recoverCase(Tbunit unit, TbCaseDataRequest data) {
        EntityManager em = (EntityManager) Component.getInstance("entityManager");

        TbCaseNG tbcase = em.find(TbCaseNG.class, data.getServerId());

        if (!tbcase.getOwnerUnit().equals(unit)) {
            throw new RuntimeException("Invlid case for the given unit: " + tbcase.getId());
        }

        return tbcase;
    }

    private TbCaseNG initNewCase(Tbunit unit, TbCaseDataRequest data) {
        TbCaseNG tbcase = new TbCaseNG();

        tbcase.setOwnerUnit(unit);
        tbcase.setNotificationUnit(unit);

        Patient patient = new Patient();
        patient.setWorkspace(UserSession.getWorkspace());

        tbcase.setPatient(patient);
        tbcase.setValidationState(ValidationState.WAITING_VALIDATION);

        return tbcase;
    }

    private EntityManager getEntityManager() {
        return (EntityManager)Component.getInstance("entityManager");
    }

    /**
     * Return an instance of {@link FieldValue} by its ID
     * @param id the field value ID
     * @return instance of {@link FieldValue}
     */
    private FieldValue getFieldValue(Integer id) {
        return id == null ? null :
                getEntityManager().find(FieldValue.class, id);
    }

    private FieldValueComponent getFieldValueComponent(Integer id) {
        return id == null ? new FieldValueComponent() :
                new FieldValueComponent(getFieldValue(id));
    }

    /**
     * Return the regimen by the given ID
     * @param id
     * @return
     */
    private Regimen getRegimenById(Integer id) {
        return id != null ?
                getEntityManager().find(Regimen.class, id) :
                null;
    }

    private AdministrativeUnit getAdminUnitById(Integer id) {
        return id != null ?
                getEntityManager().find(AdministrativeUnit.class, id) :
                null;
    }

    /**
     * Update the prescribed medicines from the list of prescribed medicines sent from the client
     * @param tbcase the TB case
     * @param prescs
     */
    private void updatePrescribedMedicines(TbCaseNG tbcase, List<PrescribedMedicineData> prescs) {
        EntityManager em = getEntityManager();
        em.createQuery("delete from PrescribedMedicine where tbcase.id = :id")
                .setParameter("id", tbcase.getId())
                .executeUpdate();

        if (tbcase.getTreatmentPeriod() == null) {
            return;
        }

        for (PrescribedMedicineData data: prescs) {
            PrescribedMedicine pm = new PrescribedMedicine();
            pm.setComments(data.getComments());
            pm.setDoseUnit(data.getDoseUnit());
            pm.setFrequency(data.getFrequency());
            pm.setSource(em.find(Source.class, data.getSourceId()));
            pm.setTbcase(tbcase);
            pm.setMedicine(em.find(Medicine.class, data.getMedicineId()));
            pm.setPeriod(new Period(data.getIniDate(), data.getEndDate()));
            tbcase.getPrescribedMedicines().add(pm);
        }
    }

    private void updateTreatmentHealthUnit(TbCase tbcase) {
        // if no treatment deletes THUs
        if (tbcase.getTreatmentPeriod() == null || tbcase.getTreatmentPeriod().isEmpty()) {
            EntityManager em = getEntityManager();
            em.createQuery("delete from TreatmentHealthUnit where tbcase.id = :id")
                    .setParameter("id", tbcase.getId())
                    .executeUpdate();
            return;
        }

        List<TreatmentHealthUnit> lst = tbcase.getSortedTreatmentHealthUnits();

        TreatmentHealthUnit hu;
        if (lst.isEmpty()) {
            hu = new TreatmentHealthUnit();
            tbcase.getHealthUnits().add(hu);
            hu.setTbcase(tbcase);
            hu.setTbunit(tbcase.getOwnerUnit());
            // set period
            Period p = new Period(tbcase.getTreatmentPeriod());
            hu.setPeriod(p);
        } else {
            hu = lst.get(lst.size() - 1);
            // set period
            Period p = new Period(hu.getPeriod().getIniDate(), tbcase.getTreatmentPeriod().getEndDate());
            hu.setPeriod(p);
        }
    }

    private void updatePrevTBTreat(TbCase tbcase, List<PrevTBTreatmentData> data) {
        EntityManager em = getEntityManager();
        em.createQuery("delete from PrevTBTreatment where tbcase.id = :id")
                .setParameter("id", tbcase.getId())
                .executeUpdate();

        if (data == null || data.size() == 0) {
            return;
        }

        // get Substances entities
        List<Substance> subList = getEntityManager().createQuery("from Substance where prevTreatmentForm = true and workspace.id = :wsId")
                .setParameter("wsId", UserSession.getWorkspace().getId())
                .getResultList();

        for (PrevTBTreatmentData item: data) {
            PrevTBTreatment prev = new PrevTBTreatment();
            prev.setMonth(item.getMonth());
            prev.setYear(item.getYear());
            prev.setOutcome(item.getOutcome());
            prev.setSubstances(new ArrayList<Substance>());
            prev.setTbcase(tbcase);

            // feed substance list
            for (Substance s : subList) {
                String propName = "has" + s.getAbbrevName().getName1().replace("/", "");
                try {
                    Boolean hasSubstance = (Boolean) PropertyUtils.getProperty(item, propName);
                    if(hasSubstance) {
                        prev.getSubstances().add(s);
                    }
                } catch (Exception e) {
                    System.out.println("Property not found in " + ExamDSTData.class.getCanonicalName() +
                            " = " + propName);
                }
            }

            tbcase.getPrevTbTreats().add(prev);
        }
    }
}
